// based on this thing by Daniel Shiffman:
// http://shiffman.net/2011/04/26/opencv-matching-faces-over-time/

#region usings
using System;
using System.ComponentModel.Composition;
using System.Collections.Generic;
using VVVV.PluginInterfaces.V1;
using VVVV.PluginInterfaces.V2;
using VVVV.Utils.VColor;
using VVVV.Utils.VMath;

using VVVV.Core.Logging;
#endregion usings

namespace VVVV.Nodes
{
	#region Rectangle
	public class Rectangle
	{
		public double x {get; set;}
		public double y {get; set;}
		public double angle {get; set;}
		public int id;
		public Vector2D pos
		{
			get
			{
				return new Vector2D(x,y);
			}			
		}
		
		// Constructor
		public Rectangle(double _x, double _y, double _angle , int _id)
		{
			x = _x;
			y = _y;
			angle = _angle;
			id = _id;
		}
		
		// Constructor
		public Rectangle(Vector2D v, double _angle, int _id)
		{
			x = v.x;
			y = v.y;
			angle = _angle;
			id = _id;
		}
		
	}
	#endregion Rectangle
	
	#region Tracker
	public class Tracker
	{	
		// Dimensions
		public Rectangle r;
		
		// Target (needet for Smoothing)
		public Rectangle target {get; set;}
		
		// Search Radius
		public double searchRadius {get; set;}
		
		// Am I available to be matched?
		public bool available;
		
		// Should I be deleted?
		public bool delete;
		
		// How long should I live if I have disappeared?
		public int timer = 127;
		public int maxLifetime = 127;
		
		// Assign a number to each person
		public int id;
		
		// Constructor
		public Tracker(double x, double y, double angle, int _id, int _timer, int _maxLifetime)
		{
			r = new Rectangle(x, y, angle, _id);
			target = r;
			available = true;
			delete = false;
			id = _id;
			timer = _timer;
			maxLifetime = _maxLifetime;
		}
		
		// Constructor
		public Tracker(Rectangle _r, int _timer, int _maxLifetime)
		{
			r = _r;
			target = _r;
			available = true;
			delete = false;
			id = _r.id;
			timer = _timer;
			maxLifetime = _maxLifetime;
		}
		
		public void update(Rectangle newTarget)
		{
			target = newTarget;
			if (timer < maxLifetime) timer++;
		}
		
		public void countDown()
		{
			timer--;
		}
		
		public bool dead()
		{
			if (timer < 0) return true;
			return false;
		}
		
		public void run(double smoothing, double _searchRadius)
		{
			searchRadius = _searchRadius;
			
			// smooth movement to Target
			r.x = smoothing * r.x + (1.0 - smoothing) * target.x;
			r.y = smoothing * r.y + (1.0 - smoothing) * target.y;
			
			// smooth rotation
			double d1 = Math.Abs(target.angle - r.angle);
			double d2 = Math.Abs((target.angle + 1) - r.angle); // overflow
			double d3 = Math.Abs((target.angle - 1) - r.angle); // underflow
			
			if(d1 > d2)
			{
				r.angle = (smoothing * r.angle + (1.0 -smoothing) * (target.angle + 1))%1;
			}
			else if(d1 > d3)
			{
				r.angle = (smoothing * (r.angle + 1) + (1.0 -smoothing) * target.angle )%1;
			}
			else
			{
				r.angle = smoothing * r.angle + (1.0 -smoothing) * target.angle;
			}
					
		}
	}
	#endregion Tracker
	
	#region PluginInfo
	[PluginInfo(Name = "Tracker",
	Category = "Animation",
	Version = "ID",
	Author = "sebl",
	Credits = "Daniel Shiffman",
	Help = "Tracks 'objects' (with given IDs)",
	Tags = "detect, ID, blob, tracking")]
	#endregion PluginInfo
	public class AnimationIDTrackerNode : IPluginEvaluate
	{
		#region fields & pins
		[Input("Pos", DefaultValues = new double[] { 0.0, 0.0 } )]
		public ISpread<Vector2D> FInPos;
		
		[Input("Rotation")]
		public ISpread<double> FInRotation;
		
		[Input("ID", DefaultValues = new double[] { 0.2, 0.6 })]
		public ISpread<int> FInID;
		
		[Input("initial Timer", DefaultValue = 50)]
		public ISpread<int> FInTimer;
		
		[Input("Max Lifetime", DefaultValue = 200)]
		public ISpread<int> FInMaxLifetime;
		
		[Input("Smoothing", IsSingle = true, MinValue = 0.0, MaxValue = 1.0, DefaultValue = 0.9)]
		public ISpread<double> FInSMoothing;
		
		[Input("Search Radius", IsSingle = true, MinValue = 0.01, DefaultValue = 1.0)]
		public ISpread<double> FInSearchRadius;
		
		[Input("Reset Counter", IsSingle = true, IsBang = true)]
		public ISpread<bool> FInResetCounter;
		
		
		[Output("Pos")]
		public ISpread<Vector2D> FOutPos;
		
		[Output("Rotation")]
		public ISpread<double> FOutRotation;
		
		[Output("Search Radius")]
		public ISpread<double> FOutSearchRadius;
		
		[Output("id")]
		public ISpread<int> FOutId;
		
		[Output("Timer")]
		public ISpread<int> FOutTimer;
		
		[Output("Available")]
		public ISpread<bool> FOutAvailable;
		
		[Output("FaceCounter")]
		public ISpread<int> FOutFaceCount;
		
		public List<Tracker> trackerList = new List<Tracker>();
		public int trackCount = 0;
		
		public List<Rectangle> rects;
		
		#endregion fields & pins
		
		#region Evaluate
		public void Evaluate(int SpreadMax)
		{
			if (FInResetCounter[0])
			{
				trackCount = 0;
				trackerList.Clear();
			}
			
			rects = new List<Rectangle>(SpreadMax);
			for (int s = 0; s < SpreadMax; s++)
			{
				rects.Add(new Rectangle(FInPos[s], FInRotation[s], FInID[s]));
			}
			
			// calculate the tracking stuff
			track(rects, FInTimer as Spread<int>, FInMaxLifetime as Spread<int>);
			
			// run all the Trackers (lerp to target)
			foreach (Tracker t in trackerList)
			{
				t.run(FInSMoothing[0], FInSearchRadius[0]);
			}
			
			// write values to Output Pins
			int numTrackers = trackerList.Count;
			
			FOutPos.SliceCount = numTrackers;
			FOutRotation.SliceCount = numTrackers;
			FOutSearchRadius.SliceCount = numTrackers;
			FOutId.SliceCount = numTrackers;
			FOutTimer.SliceCount = numTrackers;
			FOutAvailable.SliceCount = numTrackers;
			
			for (int i = 0; i < numTrackers; i++)
			{
				FOutPos[i] = trackerList[i].r.pos;
				FOutRotation[i] = trackerList[i].r.angle;
				FOutSearchRadius[i] = trackerList[i].searchRadius;
				FOutId[i] = trackerList[i].id;
				FOutTimer[i] = trackerList[i].timer;
				FOutAvailable[i] = trackerList[i].available;
			}
			
			FOutFaceCount.SliceCount = 1;
			FOutFaceCount[0] = trackCount;
		}
		#endregion Evaluate
		
		
		public static double CalculateDistance(Vector2D location1, Vector2D location2)
		{
			return (location1 - location2).Length;
		}
		
		
		private void track(List<Rectangle> rects, Spread<int> timers, Spread<int> maxLifetimes)
		{
			// -------------------------------------------------------------------------------------------
			// SCENARIO 1: trackerList is empty
			if (trackerList.Count == 0)
			{
				// Just make a Tracker object for every Rectangle
				for (int i = 0; i < rects.Count; i++)
				{
					trackerList.Add(new Tracker(rects[i], timers[i], maxLifetimes[i]));
					trackCount++;
				}
			}
			
			
			// -------------------------------------------------------------------------------------------
			// SCENARIO 2: We have fewer Tracker objects than Rectangles from the plugin Input
			else if (trackerList.Count <= rects.Count)
			{
				bool[] used = new bool[rects.Count];
				
				// Match existing Tracker objects with a Rectangle
				foreach (Tracker tracker in trackerList)
				{
					// Find rects[index] that is closest to Tracker tracker
					// set used[index] to true so that it can't be used twice
					int index = -1;
					
					for (int i = 0; i < rects.Count; i++)
					{
						// check if id matches
						bool match = false;
						if (tracker.id == rects[i].id)
							match = true;
						
						double distance = CalculateDistance(rects[i].pos, tracker.r.pos);
						
						if (match && !used[i] && distance < tracker.searchRadius)
						{
							index = i;
						}
						
					}
					// Update Tracker object location
					if (index != -1) // matching rect found
					{
						used[index] = true;
						tracker.update(rects[index]);
					}

				}
				// Add any unused rects
				for (int i = 0; i < rects.Count; i++)
				{
					if (!used[i])
					{
						trackerList.Add(new Tracker(rects[i], timers[i], maxLifetimes[i]));
						trackCount++;
					}
				}
			}
			
			
			// -------------------------------------------------------------------------------------------
			// SCENARIO 3: We have more Tracker objects than Rectangles from the plugin Input
			else
			{
				// All Tracker objects start out as available
				foreach (Tracker tracker in trackerList)
				{
					tracker.available = true;
				}
				
				// if there's no match, we have to add new Trackers in this senario, too
				bool[] used = new bool[rects.Count];
				
				// Match Rectangle with a Tracker object
				for (int i = 0; i < rects.Count; i++)
				{				
					// Find Tracker object closest to rects[i] Rectangle
					// set available to false
					int index = -1;
					for (int j = 0; j < trackerList.Count; j++)
					{
						Tracker f = trackerList[j];

						// check if id matches
						bool match = false;
						if (f.r.id == rects[i].id)
							match = true;
						
						double distance = CalculateDistance(rects[i].pos, f.r.pos);
						
						if (match && f.available && distance < f.searchRadius)
						{
							index = j;
						}
						

					}
					// Update Tracker object location
					if (index == -1) // no matching rect found
					{
						// add a new one, because none of the existing matches
						used[i] = false;
					}
					else
					{
						Tracker t = trackerList[index];
						t.available = false;
						t.update(rects[i]);
						used[i] = true;
						
					}
				}
				
				// Add any unused rects (if no match was found, as in scenario 2)
				for (int i = 0; i < rects.Count; i++)
				{
					if (!used[i])
					{
						trackerList.Add(new Tracker(rects[i], timers[i], maxLifetimes[i]));
						trackCount++;
					}
				}				
				
				// Start to kill any left over Tracker objects
				foreach (Tracker tracker in trackerList)
				{
					if (tracker.available)
					{
						tracker.countDown();
						if (tracker.dead())
						{
							tracker.delete = true;
						}
					}
				}
				
				// Delete any Tracker that should be deleted
				for (int i = trackerList.Count-1; i >= 0; i--) 
				{
					Tracker t = trackerList[i];
					if (t.delete)
					{
						trackerList.RemoveAt(i);
					}
				}
			}
			
			
		} // end track();
		
		
		
	}
}
